/********************************************************************************************************
 * QRNA - Comparative analysis of biological sequences 
 *         with pair hidden Markov models, pair stochastic context-free
 *        grammars, and probabilistic evolutionary  models.
 *       
 * Version 2.0.0 (JUN 2003)
 *
 * Copyright (C) 2000-2003 Howard Hughes Medical Institute/Washington University School of Medicine
 * All Rights Reserved
 * 
 *     This source code is distributed under the terms of the
 *     GNU General Public License. See the files COPYING and LICENSE
 *     for details.
 ***********************************************************************************************************/

/* cfg.h
 * header file for the SCFG used in the RNA model.
 * Defines structures and macros.
 *
 * ER, Mon Jun 21 12:00:21 CDT 1999 [STL]
 *
 * 
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include "squid.h"
#include "globals.h"

/* DP non-terminals 
 */
#define V       0
#define W       1
#define WB      2
#define IS1     3
#define IS2B    4
#define IS2I    5
#define STEM    6
#define NDPS    7

/* The dynamic programming matrix keeps different cells
 * for the different main types of transitions in the grammar.
 */
/* NODES that appear in V
 */
#define dpcS1          0		/* IS1-Hairpin loops ((d < MAXRNALOOP)        */
#define dpcS2S         1		/* IS2-stem      [d1 == 1, d2 == 1]           */
#define dpcS2B         2		/* IS2-bulge                                  */
#define dpcS2I         3		/* IS2-int loops [d1 >  1, d2 >  1]           */
#define dpcMV          4		/* nested multiloops (IS > 2)                 */
#define dpcEV          5		/* dummy transition                           */

/* NODES that appear in W, WB
 */
#define dpcL           6		/* leftwise transition                        */
#define dpcR           7		/* rightwise transition                       */
#define dpcP           8		/* pairwise transition                        */
#define dpcBW          9		/* normal bifurcation                         */
#define dpcEW         10		/* dummy transition                           */

/* NODES that appear in IS1, IS2B, IS2I
 */
#define dpcLEN(size)  11+(size)        	/* lenght (< MAXRNALOOP) of IS's              */
#define dpcEIS        11+MAXRNALOOP     /* dummy transition                           */

/* NODES that appear in STEM
 */
#define dpcSTEMLEN(size)  12+MAXRNALOOP+(size)        	/* lenght (< MAXRNASTEM) of IS's              */
#define dpcESTEM          12+MAXRNALOOP+MAXRNASTEM       /* dummy transition                           */

#define NNODES            13+MAXRNALOOP+MAXRNASTEM     /* # of DP cells kept                         */


extern int   DpNodeidx[NDPS];           /* node indexing for each dp matrix           */
extern int   Idx[NDPS];                 /* number of "real" states for each state     */
extern int   NodesPerDp[NDPS];          /* number of nodes for each dp matrix         */
extern int   Ntrans[NDPS];              /* total number of transitions per dp         */
extern int   TransPerDp[NDPS];          /* number of transitions per dp matrix        */
extern char *stNAME[NDPS];              /* ASCII dp matrix names                      */

/* These macros define the layout of the transition matrices.
 */
/* INDEXING for a single state V_{a b}
 */
#define idxS1st(symi,symj)          (4*symi+symj)           /* 4^2 hairpins (includes mismatches) */
#define idxS2Sst(symi,symj)         (16+4*symi+symj)        /* 4^2 IS2-stem                       */
#define idxS2Bst(symi,symj)         (32+4*symi+symj)        /* 4^2 IS2-bulge L1                   */
#define idxS2Ist(symi,symj)         (48+4*symi+symj)        /* 4^2 IS2-int loops                  */
#define idxMVst                      64                     /*   1 nested multiloop               */
#define NTRANSVst                    65                     /* total number of stV transition     */

/* index all 16 V states together / (i,j) is the new emission (if any) , (k,l) the state we come from 
 */
#define idxS1(symi,symj,symk,syml)  (idxS1st(symi,symj)  + (4*symk+syml)*NTRANSVst)   /* 4^4 prob */
#define idxS2S(symi,symj,symk,syml) (idxS2Sst(symi,symj) + (4*symk+syml)*NTRANSVst)   /* 4^4 prob */
#define idxS2B(symi,symj,symk,syml) (idxS2Bst(symi,symj) + (4*symk+syml)*NTRANSVst)   /* 4^4 prob */  
#define idxS2I(symi,symj,symk,syml) (idxS2Ist(symi,symj) + (4*symk+syml)*NTRANSVst)   /* 4^4 prob */  
#define idxMV(symk,syml)            (idxMVst             + (4*symk+syml)*NTRANSVst)   /* 4^2 prob */ 
#define idxEV                       (16*NTRANSVst)                /*           1 empty transition */ 
#define NTRANSV                     (16*NTRANSVst+1)              /* total number of V transition */

/* INDEXING for W, WB
 */
#define idxL(symi)                  (symi)                  /*   4 leftwise                       */
#define idxR(symi)                  (4+symi)                /*   4 rightwise                      */
#define idxP(symi,symj)             (8+4*symi+symj)         /* 4^2 pairwise transitions           */
#define idxBW                        24	                    /*   1 normal bifurcation             */
#define idxEW                        25                     /*   1 empty transition               */
#define NTRANSW                      26                     /* total number of W transition       */

/* INDEXING for IS1, IS2
 */
#define idxLEN(size)                (size)                  /* emits everything                   */
#define idxEIS                      (MAXRNALOOP)            /* 1          empty transition        */
#define NTRANSIS                    (MAXRNALOOP+1)          /* total number of IS transition      */

/* INDEXING for STEM
 */
#define idxSTEMLEN(size)            (size)                  /* emits everything                   */
#define idxESTEM                    (MAXRNASTEM)            /* 1          empty transition        */
#define NTRANSSTEM                  (MAXRNASTEM+1)          /* total number of IS transition      */

/* NODES that appear in S
 */
#define dpcPS     0   /* S->aSb */
#define dpcLS     1   /* S->aL  */
#define dpcRS     2   /* S->Ra  */
#define dpcBS     3   /* S->LS  */
#define dpcES     4   /* S->aeb  */

/* NODES that appear in L
 */
#define dpcPL     5   /* L->aSb */
#define dpcLL     6   /* L->aL  */
#define dpcEL     7   /* L->aeb  */

/* NODES that appear in R
 */
#define dpcRR     8   /* R->Ra */
#define dpcER     9   /* R->e  */

/* NODES total
 */
#define NNODESNR 10  

/* Structure: trace_s
 * 
 * Binary tree structure for storing an RNA structure,
 * derived as a traceback of an alignment of SCFG to sequence.
 */
struct trace_s {
  int emitl;			/* i position (0..N-1) or -1 if nothing   */
  int emitr;			/* j position (0..N-1) or -1 if nothing   */

  int type;                     /* (i,j) state */
    
  struct trace_s *nxtl;		/* ptr to left (or only) branch, or NULL for end    */
  struct trace_s *nxtr;		/* ptr to right branch, for o(mx)^2 only, else NULL */
  struct trace_s *prv;          /* ptr to parent                                    */
};

/* Structure: tracekn_s
 * 
 * Binary tree structure for storing an RNA structure,
 * derived as a traceback of an alignment of SCFG to sequence.
 */
struct tracekn_s {
  int emiti;                    /* i position (0..N-1) or -1 if nothing   */
  int emitj;                    /* j position (0..N-1) or -1 if nothing   */
  int emitk;                    /* k position (0..N-1) or -1 if nothing   */
  int emitl;                    /* l position (0..N-1) or -1 if nothing   */
 
  int type;                    /* (i,j) status : dpcP (paired), dcpL (unparied) */
  int node;                     /* type of transition node                */                                                                            
  struct tracekn_s *nxtl;       /* ptr to left (or only) branch, or NULL for end    */
  struct tracekn_s *nxtr;       /* ptr to right branch, for o(mx)^2 only, else NULL */
  struct tracekn_s *prv;        /* ptr to parent                                    */
};
  
/* Structure: tracestack_s
 *
 * A pushdown stack used for traversing a binary tree of trace_s structures.
 */
struct tracestack_s {
  struct trace_s      *node;
  struct tracestack_s *nxt;
};

/* Structure: traceknstack_s
 *
 * A pushdown stack used for traversing a binary tree of trace_s structures.
 */
struct traceknstack_s {
  struct tracekn_s      *node;
  struct traceknstack_s *nxt;
};


/* Structure: tracer_s
 * 
 * Binary tree structure for storing an alignment,
 * derived as a traceback of an alignment of regular grammar to sequence.
 */
struct tracer_s {
  int emit;			/* i position (0..N-1) or -1 if nothing   */

  int type;                     /*  state */
    
  struct tracer_s *nxt;		/* ptr to child , or NULL for end    */
  struct tracer_s *prv;          /* ptr to parent                     */
};

/* Structure: tracerstack_s
 *
 * A pushdown stack used for traversing a linear tree of tracer_s structures.
 */
struct tracerstack_s {
  struct tracer_s      *node;
  struct tracerstack_s *nxt;
};






